<?php
include("connection.php");
include("jwt.php");

// Allow CORS
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

header('Content-Type: application/json');

function sendResponse($code, $msg = '', $payload = '')
{
    http_response_code($code);
    echo json_encode([
        'code' => $code,
        'msg' => $msg,
        'payload' => $payload
    ]);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // --- Authentication ---
    $headers = getallheaders();
    $auth_header = $headers['Authorization'] ?? '';

    if (!$auth_header || !str_starts_with($auth_header, 'Bearer ')) {
        sendResponse(401, 'Authorization header missing');
        exit;
    }

    $token = str_replace('Bearer ', '', $auth_header);
    $validation = json_decode(is_jwt_valid($token), true);

    if (!isset($validation['status']) || $validation['status'] !== 'Success') {
        sendResponse(403, 'Invalid or expired token');
        exit;
    }

    // --- Input Validation ---
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['booking_id']) || !isset($data['mark'])) {
        sendResponse(400, 'Booking ID and Mark are required');
        exit;
    }

    $booking_id = $data['booking_id'];
    $mark = $data['mark'];

    // Validate the 'mark' value is one of the allowed enum values ["0", "1", "2"]
    $validMarks = ["0", "1", "2"];
    if (!in_array($mark, $validMarks)) {
        sendResponse(400, 'Invalid value for mark. Allowed values are: "0", "1", or "2".');
        exit;
    }

    // --- Update Query ---
    $query = "UPDATE bookings SET mark = ? WHERE id = ?";
    $stmt = $conn->prepare($query);
    
    if ($stmt) {
        $stmt->bind_param("si", $mark, $booking_id); // 'si' for string and integer
        $stmt->execute();

        if ($stmt->affected_rows > 0) {
            sendResponse(200, 'Booking updated successfully');
        } else {
            sendResponse(404, 'Booking not found or no changes made');
        }

        $stmt->close();
    } else {
        sendResponse(500, 'Database error: ' . $conn->error);
    }
} else {
    sendResponse(405, "Method Not Allowed");
}
?>
